﻿namespace Microsoft.Samples.PlanMyNight.Data.Caching.Tests
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Runtime.Serialization.Json;
    using System.Text;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;

    [TestClass]
    public class CachedActivitiesRepositoryFixture
    {
        [TestMethod]
        public void RetrieveActivityFromPrimedCache()
        {
            var activityId = ToBase64("NYX3333|1|55~-72");

            var sourceRepo = new Mock<IActivitiesRepository>();
            sourceRepo.Setup(o => o.RetrieveActivity(activityId))
                      .Returns(new Activity { Id = activityId, BingId = "NYX3333" });

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Activity", "NYX3333")).Returns(default(Activity));

            var cache = new CachedActivitiesRepository(cacheProvider.Object, sourceRepo.Object);
            var result = cache.RetrieveActivity(activityId);

            Assert.IsNotNull(result);
            sourceRepo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void RetrieveActivityFromCache()
        {
            var activityId = ToBase64("NYX3333|1|55~-72");

            var sourceRepo = new Mock<IActivitiesRepository>();

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Activity", "NYX3333")).Returns(new Activity { Id = activityId, BingId = "NYX3333" });

            var cache = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            var result = cache.RetrieveActivity(activityId);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void SearchFromPrimedCacheAlsoCachesActivities()
        {
            var searchQuery = new AdvancedSearchQuery
            {
                State = "WA",
                City = "Colville",
                ActivityTypeId = 3
            };

            var cacheKey = Serialize(searchQuery);

            var items = new Activity[] {
                new Activity { BingId = "NYX3333" },
                new Activity { BingId = "NYX4444" },
                new Activity { BingId = "NYX5555" }
            };

            var sourceRepo = new Mock<IActivitiesRepository>();
            sourceRepo.Setup(o => o.Search(It.IsAny<AdvancedSearchQuery>()))
                      .Returns(new PagingResult<Activity>(items));

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ActivitySearch", cacheKey)).Returns(null);

            cacheProvider.Setup(o => o.HasKey("Activity", "NYX3333")).Returns(false);
            cacheProvider.Setup(o => o.Add("Activity", "NYX3333", It.IsAny<Activity>(), It.IsAny<TimeSpan>()));

            cacheProvider.Setup(o => o.HasKey("Activity", "NYX4444")).Returns(false);
            cacheProvider.Setup(o => o.Add("Activity", "NYX4444", It.IsAny<Activity>(), It.IsAny<TimeSpan>()));

            cacheProvider.Setup(o => o.HasKey("Activity", "NYX5555")).Returns(true);

            var cache = new CachedActivitiesRepository(cacheProvider.Object, sourceRepo.Object);
            var result = cache.Search(searchQuery);

            Assert.IsNotNull(result);
            sourceRepo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void SearchFromCache()
        {
            var searchQuery = new AdvancedSearchQuery
            {
                State = "WA",
                City = "Colville",
                ActivityTypeId = 3
            };

            var cacheKey = Serialize(searchQuery);

            var items = new Activity[] {
                new Activity { BingId = "NYX3333" },
                new Activity { BingId = "NYX4444" },
                new Activity { BingId = "NYX5555" }
            };

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ActivitySearch", cacheKey)).Returns(new PagingResult<Activity>(items));

            var cache = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            var result = cache.Search(searchQuery);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void NaturalSearchPrimedCacheCallsRepository()
        {
            var query = new NaturalSearchQuery { Query = "bars in manhattan, ny", ActivityTypeId = 1, Page = 1, PageSize = 10, Type = SearchType.Itinerary };
            var cacheKey = Serialize(query);

            var items = new Activity[] {
                new Activity { BingId = "NYX3333" },
                new Activity { BingId = "NYX4444" },
                new Activity { BingId = "NYX5555" }
            };

            var sourceRepo = new Mock<IActivitiesRepository>();
            sourceRepo.Setup(o => o.Search(It.IsAny<NaturalSearchQuery>())).Returns(new PagingResult<Activity>(items));

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ActivitySearch", cacheKey)).Returns(null);
            cacheProvider.Setup(o => o.Add("ActivitySearch", cacheKey, It.IsAny<PagingResult<Activity>>(), It.IsAny<TimeSpan>()));

            cacheProvider.Setup(o => o.HasKey("Activity", "NYX3333")).Returns(false);
            cacheProvider.Setup(o => o.Add("Activity", "NYX3333", It.IsAny<Activity>(), It.IsAny<TimeSpan>()));

            cacheProvider.Setup(o => o.HasKey("Activity", "NYX4444")).Returns(false);
            cacheProvider.Setup(o => o.Add("Activity", "NYX4444", It.IsAny<Activity>(), It.IsAny<TimeSpan>()));

            cacheProvider.Setup(o => o.HasKey("Activity", "NYX5555")).Returns(true);

            var cachedRepo = new CachedActivitiesRepository(cacheProvider.Object, sourceRepo.Object);
            var result = cachedRepo.Search(query);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
            sourceRepo.VerifyAll();
        }

        [TestMethod]
        public void NaturalSearchRetrievesFromCache()
        {
            var query = new NaturalSearchQuery { Query = "bars in manhattan, ny", ActivityTypeId = 1, Page = 1, PageSize = 10, Type = SearchType.Itinerary };
            var cacheKey = Serialize(query);

            var items = new Activity[] {
                new Activity { BingId = "NYX3333" },
                new Activity { BingId = "NYX4444" },
                new Activity { BingId = "NYX5555" }
            };

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ActivitySearch", cacheKey)).Returns(new PagingResult<Activity>(items));

            var cachedRepo = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            var result = cachedRepo.Search(query);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void GeocodeAddressPrimedCache()
        {
            var address = new ActivityAddress
            {
                City = "New York",
                State = "NY",
                StreetAddress = "322 5th Avenue"
            };

            var cacheKey = Serialize(address);

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Geocode", cacheKey)).Returns(null);
            cacheProvider.Setup(o => o.Add("Geocode", cacheKey, It.IsAny<Tuple<double, double>>(), It.IsAny<TimeSpan>()));

            var sourceRepo = new Mock<IActivitiesRepository>();
            sourceRepo.Setup(o => o.GeocodeAddress(address)).Returns(new Tuple<double, double>(0, 0));

            var cache = new CachedActivitiesRepository(cacheProvider.Object, sourceRepo.Object);
            var result = cache.GeocodeAddress(address);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
            sourceRepo.VerifyAll();
        }

        [TestMethod]
        public void GeocodeAddressFromCache()
        {
            var address = new ActivityAddress
            {
                City = "New York",
                State = "NY",
                StreetAddress = "322 5th Avenue"
            };

            var cacheKey = Serialize(address);

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Geocode", cacheKey)).Returns(new Tuple<double, double>(0, 0));

            var cache = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            var result = cache.GeocodeAddress(address);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void ParseQueryLocationPrimedCache()
        {
            var sourceRepo = new Mock<IActivitiesRepository>();
            sourceRepo.Setup(o => o.ParseQueryLocation("bars in manhattan, ny"))
                      .Returns(new ActivityAddress { State = "New York", City = "Manhattan" });

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Geocode", "BARS IN MANHATTAN, NY")).Returns(null);
            cacheProvider.Setup(o => o.Add("Geocode", "BARS IN MANHATTAN, NY", It.IsAny<ActivityAddress>(), It.IsAny<TimeSpan>()));

            var cachedRepo = new CachedActivitiesRepository(cacheProvider.Object, sourceRepo.Object);
            var result = cachedRepo.ParseQueryLocation("bars in manhattan, ny");

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
            sourceRepo.VerifyAll();
        }

        [TestMethod]
        public void ParseQueryLocationFromCache()
        {
            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Geocode", "BARS IN MANHATTAN, NY")).Returns(new ActivityAddress { State = "New York", City = "Manhattan" });

            var cachedRepo = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            var result = cachedRepo.ParseQueryLocation("bars in manhattan, ny");

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void PopulateItineraryActivitiesCallsItSelf()
        {
            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Activity", "NYX1111")).Returns(new Activity());
            cacheProvider.Setup(o => o.Get("Activity", "NYX2222")).Returns(new Activity());
            cacheProvider.Setup(o => o.Get("Activity", "NYX3333")).Returns(new Activity());

            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = ToBase64("NYX1111|1|55~-72") });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = ToBase64("NYX2222|1|55~-72") });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = ToBase64("NYX3333|1|55~-72") });

            var cache = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            cache.PopulateItineraryActivities(itinerary);

            foreach (var item in itinerary.Activities)
            {
                Assert.IsNotNull(item.Activity);
            }

            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void RetrieveActivityTypesFromPrimedCache()
        {
            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Reference", "activityTypes")).Returns(null);
            cacheProvider.Setup(o => o.Add("Reference", "activityTypes", It.IsAny<IEnumerable<ActivityType>>(), It.IsAny<TimeSpan>()));

            var sourceRepo = new Mock<IActivitiesRepository>();
            sourceRepo.Setup(o => o.RetrieveActivityTypes()).Returns(new ActivityType[0]);

            var cache = new CachedActivitiesRepository(cacheProvider.Object, sourceRepo.Object);
            var result = cache.RetrieveActivityTypes();

            Assert.IsNotNull(result);
            sourceRepo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void RetrieveActivityTypesFromCache()
        {
            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Reference", "activityTypes")).Returns(new ActivityType[0]);

            var cache = new CachedActivitiesRepository(cacheProvider.Object, new Mock<IActivitiesRepository>().Object);
            var result = cache.RetrieveActivityTypes();

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        private static string ToBase64(string key)
        {
            return Convert.ToBase64String(Encoding.Default.GetBytes(key));
        }

        private static string Serialize<T>(T keyObject)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new DataContractJsonSerializer(typeof(T));
                serializer.WriteObject(stream, keyObject);
                return Encoding.Default.GetString(stream.ToArray());
            }
        }
    }
}
